/*********************************************************************
*
*   RobotBase.h - Robot base class
*
*   COPYRIGHT:
*       Software source code by Alex Morozov and Chris D. Locke is
*       licensed under a Creative Commons Attribution-Noncommercial-
*       Share Alike 3.0 United States License
*       (http://creativecommons.org/licenses/by-nc-sa/3.0/us/)
*
*********************************************************************/
#pragma once

#include <stdio.h>
#include "rob_pub.h"

#include "RobotMath.h"
#include "Watchdog.h"

class RobotBase 
{
// Public constants
public:
    typedef enum
    {
        // enumerate line following sensors
        LFS_LEFT = 0,
        LFS_CENTER = 1,
        LFS_RIGHT = 2,
        LFS_CNT = 3
    };
    
private:                            // private variables
    double m_leftMotor;
    double m_rightMotor;
    double m_x;
    double m_y;
    double m_yaw;                   // robot heading in radians
                                    // measured clockwise from the North
    double m_width;                 // robot width
    FILE *m_logFile;                // log file
    Watchdog            m_watchdog; // robot's watchdog
    rob_mode_type       m_mode;     // current robot mode
    unsigned int m_update_rate_ms;  // update rate in milliseconds
    PositionVector      m_lfs_pos_bc[ LFS_CNT ];
                                    // line following sensor positions in the body coordinate system
    PositionVector      m_lfs_pos_fc[ LFS_CNT ];
                                    // line following sensor positions in the field coordinate system

public:
    const PositionVector& GetLfsPosFc( int i ) const
    {
        _ASSERT( i < LFS_CNT );
        return( m_lfs_pos_fc[ i ] );
    }
    
    const PositionVector& GetLfsPosBc( int i ) const
    {
        _ASSERT( i < LFS_CNT );
        return( m_lfs_pos_bc[ i ] );
    }

    /*
     *  GetRobotPosition()
     *  Outputs current robot position and heading (in degrees).
     */
    void GetRobotPosition( float * x, float * y, float * yaw )
    {
        *x   = ( float )m_x;
        *y   = ( float )m_y;
        *yaw = ( float )( m_yaw * 180.0 / PI );
    };

    /*
     * GetMode()
     * Returns current robot mode
     */
    rob_mode_type GetMode( void ) { return m_mode; }

    /*
     *  GetMotorSpeed()
     *  Outputs current motor setting as an integer between -100 and 100
     *  (inclusive)
     */
    void GetMotorSpeed( int * left, int * right )
    {
        *left = ( int )( m_leftMotor * 100 );
        *right = ( int )( m_rightMotor * 100 );
    }

    /*
     *  SetMode()
     *  Set operational mode (disabled, autonomous, periodic)
     */
    void SetMode( rob_mode_type mode );


    /*
     *  SetLeftMotor()
     *  Set left motor speed.
     *   Valid speed is [-1.0,1.0]
     */
    void SetLeftMotor( double speed )
    {
        SetMotor( &m_leftMotor, speed );
    };

    /*
     *  SetRightMotor()
     *  Set right motor speed.
     *  Valid speed is [-1.0,1.0]
     */
    void SetRightMotor( double speed )
    {
        SetMotor( &m_rightMotor, speed );
    };

    /*
     * Update robot simulation
     */
    void Update
        ( 
        unsigned int period_ms      // update rate in milliseconds
        );
        

    /*
     * The following functions should be implemented by the derived classes
     */
	virtual void RobotInit() = 0;
	virtual void DisabledInit() = 0;
	virtual void AutonomousInit() = 0;
	virtual void TeleopInit() = 0;
	virtual void DisabledPeriodic() = 0;
	virtual void AutonomousPeriodic() = 0;
	virtual void TeleopPeriodic() = 0;


    static RobotBase * p_robot;     // pointer to the last created robot instance

    // The following variables are used to simulate quad encoders
    double leftMotorAccumulator;
    double rightMotorAccumulator;

protected:
    RobotBase( );
    ~RobotBase( );
    
    /**************************************************************************
    * GetLoopsPerSec()
    *   How many times per second is the periodic function called?
    **************************************************************************/
    unsigned int GetLoopsPerSec( void )
    {
        return( 1000 / m_update_rate_ms );
    }

    /**************************************************************************
    * GetWatchdog()
    *   Returns reference to the robot's watchdog
    **************************************************************************/
    Watchdog& GetWatchdog( );

    void TestPeriodic( unsigned int t_ms );

    void MoveRobot( unsigned int period_ms );
    void SetMotor( double * p_motor, double speed );
    
// Private member functions
private:
    // Update line following sensor position
    void UpdateLFSPos( void )
    {
        RotationMatrix r( m_yaw );
        for( int i = 0; i < LFS_CNT; i++ )
        {
            m_lfs_pos_fc[ i ].multiply( r, m_lfs_pos_bc[ i ] );
            m_lfs_pos_fc[ i ].element( PositionVector::X ) += m_x;
            m_lfs_pos_fc[ i ].element( PositionVector::Y ) += m_y;
        }
    }
};

#define START_ROBOT_CLASS( class_name ) \
    static class_name robot;

