/*********************************************************************
*
*   RobotMath.h - Robot math header
*
*   Copyright:
*       Software source code by Alex Morozov and Chris D. Locke is
*       licensed under a Creative Commons Attribution-Noncommercial-
*       Share Alike 3.0 United States License
*       (http://creativecommons.org/licenses/by-nc-sa/3.0/us/)
*
*********************************************************************/
#pragma once

#include <math.h>

#include "CppBot_types.h"

#define EPSILON             ( 0.0001 )
#define PI                  ( 3.14159265 )
#define PI_F                float( PI )

void ComputeRotationAngle( double leftWheelSpeed,
                           double rightWheelSpeed,
                           double robotWidth,
                           double * angle,
                           double * distance );
double ConvertYawToTheta( double yaw );

void RotateAboutPoint( double * x, double * y, double xo, double yo, double angle );

class Matrix
{
protected:
    double * m_elements;            // matrix elements
    int m_n_cols;                   // number of columns
    int m_n_rows;                   // number of rows

public:
    Matrix( int n_rows, int n_cols ) : m_elements( new double[ n_rows * n_cols ] ), m_n_cols( n_cols ), m_n_rows( n_rows ) {}
    Matrix( const Matrix& other ) : 
        m_elements( new double[ other.m_n_rows * other.m_n_cols ] ), 
        m_n_cols( other.m_n_cols ), 
        m_n_rows( other.m_n_rows )
    {
        int size = m_n_cols * m_n_rows;
        for( int i = 0; i < size; i++ )
            {
            m_elements[ i ] = other.m_elements[ i ];
            }
    }
    
    ~Matrix() { delete [] m_elements; }
    double& element( int row, int col ) const { return( m_elements[ row * m_n_cols + col ] ); }
    void multiply( const Matrix& a, const Matrix &b )
    {
        _ASSERT( m_n_rows == a.m_n_rows );
        _ASSERT( a.m_n_cols == b.m_n_rows );
        _ASSERT( b.m_n_cols == m_n_cols );
        int p = a.m_n_cols;
        for( int i = 0; i < m_n_rows; i++ )
        {
            for( int j = 0; j < m_n_cols; j++ )
            {
                double sum = 0.0;
                for( int k = 0; k < p; k++ )
                {
                sum = sum + a.element( i, k ) * b.element( k, j );
                }
                
                element( i, j ) = sum;
            }
        }    
    }
};


class PositionVector : public Matrix
{
public:
    enum { X, Y, DIM };
    
public:
    double& x;
    double& y;

public:
    PositionVector() : 
        Matrix( DIM, 1 ),
        x( m_elements[ X ] ),
        y( m_elements[ Y ] )
    {
    }
    
    double& element( int dim ) const
    {
        _ASSERT( dim < DIM );
        return( m_elements[ dim ] );
    }
};


class RotationMatrix : public Matrix
{
public:
    RotationMatrix( double theta ) : Matrix( 2, 2 )
    {
        double cos_theta = cos( theta );
        double sin_theta = sin( theta );
        m_elements[ 0 ] = cos_theta;
        m_elements[ 1 ] = -sin_theta;
        m_elements[ 2 ] = sin_theta;
        m_elements[ 3 ] = cos_theta;
    }
};