/*
 *  SimInterface.h
 *  CppBot
 *
 *  Created by Alex on 1/31/10.
 *  Copyright 2010 __MyCompanyName__. All rights reserved.
 *
 */

#pragma once

#include <vector>
#include "CppBot_types.h"

#include "SimObject.h"

class SimInterface
{
public:                             // public constants    
    enum
    {
        PWM_CNT         = 9,        // total number of PWM channels
        DIO_CNT         = 13,       // total number of digital channels
        ANALOG_CNT      = 9,        // total number of analog channels
        JOYSTICK_CNT    = 4         // total number of joysticks
    };
    typedef enum { X_AXIS, Y_AXIS, Z_AXIS } joystick_axis_type;
    
private:                            // private types
    typedef struct
        {
        float x;                    /* x-axis position [-1;+1]              */
        float y;                    /* y-axis position [-1;+1]              */
        float z;                    /* z-axis position [-1;+1]              */
        unsigned int button_mask;   /* bitmask of pressed buttons           */
        } joystick_data_type;
        
    typedef std::vector< SimObject * > sim_object_vector_type;
        
private:                            // private variables

    static float m_analog[ ANALOG_CNT ];
                                    // analog channels
    static bool m_dio[ DIO_CNT ];   // digital i/o
    static double m_encoders[ PWM_CNT ];
                                    // Currently, encoder channel must match the PWM channel
                                    // TODO: make this more flexible
    static joystick_data_type m_joysticks[ JOYSTICK_CNT ];
    static float m_pwm[ PWM_CNT ];  // PWM
    static sim_object_vector_type m_objects;
                                    // simulated objects

public:                             // public functions    
    /**************************************************************************
    * Get analog value
    **************************************************************************/
    static float GetAnalog( unsigned int channel )
    {
        _ASSERT( channel < ANALOG_CNT );
        return( m_analog[ channel ] );
    }

    /**************************************************************************
    * Get digital I/O value
    **************************************************************************/
    static bool GetDIO( unsigned int channel )
    {
        _ASSERT( channel < DIO_CNT );
        return( m_dio[ channel ] );
    }
    
    /**************************************************************************
    * Get encoder value
    **************************************************************************/
    static int GetEncoder( int channel )
    {
        // Currently, encoder channel must match the PWM channel
        // TODO: fix this
        _ASSERT( channel < PWM_CNT );
        return( int( m_encoders[ channel ] ) );
    }
    
    static float GetJoystickAxis( int joystick_idx, joystick_axis_type axis_idx );
    static bool GetJoystickButton( int joystick_idx, unsigned int button_idx );
    static float GetPWM( unsigned int channel );

	static void IncrementEncoder( int channel, double increment )
	{
		_ASSERT( channel < PWM_CNT );
		m_encoders[ channel ] += increment;
	}

    static void Register( SimObject * p_sim_object );
    
    static void ResetEncoder( int channel )
    {
        // Currently, encoder channel must match the PWM channel
        // TODO: fix this
        _ASSERT( channel < PWM_CNT );
        m_encoders[ channel ] = 0;
    }
    
    /**************************************************************************
    * Set analog value
    **************************************************************************/
    static void SetAnalog( unsigned int channel, float value )
    {
        _ASSERT( channel < ANALOG_CNT );   
        m_analog[ channel ] = value;
    }

    /**************************************************************************
    * Set digital I/O value
    **************************************************************************/
    static void SetDIO
        ( 
        unsigned int channel_idx, 
        bool channel_value 
        )
    {
    _ASSERT( channel_idx < DIO_CNT );
    m_dio[ channel_idx ] = channel_value;
    }
    
    static void SetJoystick( unsigned int n, float x, float y, float z, unsigned int button_mask );
    static void SetPWM( unsigned int channel, float value );
    static void Update( unsigned int update_rate_ms );
};