/*
 *  SimRotator.cpp
 *  CppBot
 *
 *  Created by Alex on 2/1/10.
 *  Copyright 2010 __MyCompanyName__. All rights reserved.
 *
 */

#include "SimRotator.h"

#include "SimInterface.h"
#include "RobotMath.h"

/**************************************************************************
* SimRotator - constructor
**************************************************************************/
SimRotator::SimRotator
    (
    unsigned int pwm_channel,       // PWM channel (1 to maximum number of channels)
    unsigned int analog_channel,    // analog channel
    float max_rpm,                  // Maximum rotation rate (revolutions per minute)
    float zero_to_max_s,            // how many seconds it takes to go from zero to full speed
    float init_pos_deg              // Initial position (degrees, measured counterclockwise from the X-axis)
    ) :
    m_channel( pwm_channel ),
    m_analog_channel( analog_channel ),
    m_max_rate_dps( 360.0 * max_rpm / 60.0 ),
    m_rate_dps( 0.0 ),
    m_angle_deg( init_pos_deg )
{
m_drate_dt_dpss = m_max_rate_dps / zero_to_max_s;
}   /* SimRotator::SimRotator */


/**************************************************************************
* Update - Periodic update function
**************************************************************************/
void SimRotator::Update
    ( 
    unsigned int update_rate_ms // update rate in milliseconds
    )
{
    // save current angle so that we can compute analog sensor change
    double prev_angle_deg = m_angle_deg;
    
    // Compute new angle
    float dt_s = update_rate_ms / 1000.0f;
    double unfiltered_rate_dps = m_max_rate_dps * get_pwm();
    m_rate_dps = float( limit_rate( unfiltered_rate_dps, m_rate_dps, m_drate_dt_dpss * dt_s ) );
    double angle_deg = m_angle_deg + m_rate_dps * dt_s;
    m_angle_deg = limit( angle_deg );
    
    // Update analog sensor (gyro) value
    double delta_deg = m_angle_deg - prev_angle_deg;
    double delta_rad = delta_deg * PI_F / 180.0f;
    double analog_rad = SimInterface::GetAnalog( m_analog_channel );
    analog_rad = analog_rad + delta_rad;
    SimInterface::SetAnalog( m_analog_channel, float( analog_rad ) );

    // Update encoder value
    SimInterface::IncrementEncoder( m_channel, delta_deg * ENCODER_GAIN );
}   /* SimRotator::Update */


/**************************************************************************
* limit - limit the specified angle value to the allowed range and return 
*   the result
**************************************************************************/
double SimRotator::limit( double angle_deg )
{
if( angle_deg < 0.0 )
    {
    return( angle_deg + 360.0 );
    }
else if( angle_deg > 360.0 )
    {
    return( angle_deg - 360.0 );
    }
else
    {
    return( angle_deg );
    }
}   /* SimRotator::limit */
